//  Steven Kappes
//  MD2Model.cpp
//  Description: Class to load/draw models

#include "MD2Model.h"
#include <vector>
#include <math.h>
#include <iostream>
#include "Texture.h"
using namespace std;

//#define QUERY_EXTENSION_ENTRY_POINT(name, type)               \
//    name = (type)GET_PROC_ADDRESS(#name);
//PFNGLACTIVETEXTUREARBPROC glActiveTexture;

// Private structures
typedef struct {
	unsigned char v[3];
	unsigned char normalIndex;
} FramePoint_t;

typedef struct {
	float scale[3];
	float translate[3];
	char name[16];
	FramePoint_t * fp;
} Frame_t;

typedef struct {
	unsigned short meshIndex[3];
	unsigned short stIndex[3];
} Mesh_t;

typedef struct {
	float s;
	float t;
} TexCoord_t;

typedef struct {
	float x, y, z;
} ModelPoint;

typedef struct {
	int ident;
	int version;
	int skinWidth;
	int skinHeight;
	int frameSize;
	int numSkins;
	int numVertices;
	int numST;
	int numTris;
	int numGLcmds;
	int numFrames;
	int offsetSkins;
	int offsetST;
	int offsetTris;
	int offsetFrames;
	int offsetGLcmds;
	int offsetEnd;
} ModelHeader_t;

typedef struct {
	Mesh_t * triIndex;
	TexCoord_t * st;
	vector<ModelPoint> * pointList;
	unsigned int numFrames;
	int numVertices;
	int numTriangles;
	int numST;
	int frameSize;
} ModelData_t;

MD2Model::MD2Model()
{
	modelData = NULL;
	texPath = NULL;
}

MD2Model::MD2Model(char * mPath, char * tPath)
{	//ilutRenderer(ILUT_OPENGL);
	FILE * filePtr;
	filePtr = fopen(mPath, "rb");

	modelData = NULL;
	if (filePtr != NULL) {
		ModelHeader_t * header;

		header = (ModelHeader_t *)this->getHeader(filePtr);
		if (header != NULL) {
			modelData = this->getData(filePtr, (void *)header);
			free(header);
		}

		fclose(filePtr);
	}

	texPath = (char *)malloc(sizeof(char) * strlen(tPath) + 1);
	strcpy(texPath, tPath);
}

MD2Model::~MD2Model()
{
	if (modelData != NULL) {
		ModelData_t * data = (ModelData_t *)modelData;
		free(data->triIndex);
		free(data->st);
		free(modelData);
		if (texPath != NULL) free(texPath);
	}
}

unsigned int MD2Model::frameCount(void)
{
	return ((modelData == NULL) ? 0 : ((ModelData_t *)modelData)->numFrames);
}

void MD2Model::render(unsigned int frame)
{
	if (modelData == NULL) return;

	ModelPoint vector;
	ModelData_t * data = (ModelData_t *)modelData;
	float v1[3], v2[3], v3[3];
	int offset = frame * data->numVertices;
	int i;

	if (strlen(texPath) > 0)
		glBindTexture(GL_TEXTURE_2D, fetchTexture(texPath, true, true));

    glBegin(GL_TRIANGLES);
		for (i = 0; i < data->numTriangles; i++) {
			vector = data->pointList->at(offset + data->triIndex[i].meshIndex[0]);
			v1[0] = vector.x;
			v1[1] = vector.y;
			v1[2] = vector.z;
			vector = data->pointList->at(offset + data->triIndex[i].meshIndex[2]);
			v2[0] = vector.x;
			v2[1] = vector.y;
			v2[2] = vector.z;
			vector = data->pointList->at(offset + data->triIndex[i].meshIndex[1]);
			v3[0] = vector.x;
			v3[1] = vector.y;
			v3[2] = vector.z;

			// Calculate the normal of the triangle
			this->calculateNormal(v1, v2, v3);

			// Render the triangle
			glVertexAttrib1f(surfaceAttributeLoc, i);
			glTexCoord2f(data->st[data->triIndex[i].stIndex[0]].s, 1 - data->st[data->triIndex[i].stIndex[0]].t);
			glVertex3fv(v1);

			glVertexAttrib1f(surfaceAttributeLoc, i);
			glTexCoord2f(data->st[data->triIndex[i].stIndex[2]].s, 1 - data->st[data->triIndex[i].stIndex[2]].t);
			glVertex3fv(v2);

			glVertexAttrib1f(surfaceAttributeLoc, i);
			glTexCoord2f(data->st[data->triIndex[i].stIndex[1]].s, 1 - data->st[data->triIndex[i].stIndex[1]].t);
			glVertex3fv(v3);
		}
	glEnd();

	glBindTexture(GL_TEXTURE_2D, 0);
}

void MD2Model::animate(unsigned int startFrame, unsigned int endFrame, float interpolate)
{
	ModelData_t * data = (ModelData_t *)modelData;
	ModelPoint vec1, vec2;
	int offset, offset2;
	int i;
	float v1[3], v2[3], v3[3];

	if (data == NULL) return;

	if ((startFrame >= data->numFrames) || (endFrame >= data->numFrames))
		return;

	offset = startFrame * data->numVertices;
	offset2 = endFrame * data->numVertices;

	if (strlen(texPath) > 0)
		glBindTexture(GL_TEXTURE_2D, fetchTexture(texPath, true, true));

	glBegin(GL_TRIANGLES);
		for (i = 0; i < data->numTriangles; i++) {
			vec1 = data->pointList->at(offset + data->triIndex[i].meshIndex[0]);
			vec2 = data->pointList->at(offset2 + data->triIndex[i].meshIndex[0]);
			v1[0] = vec1.x + (interpolate * (vec2.x - vec1.x));
			v1[1] = vec1.y + (interpolate * (vec2.y - vec1.y));
			v1[2] = vec1.z + (interpolate * (vec2.z - vec1.z));

			vec1 = data->pointList->at(offset + data->triIndex[i].meshIndex[2]);
			vec2 = data->pointList->at(offset2 + data->triIndex[i].meshIndex[2]);
			v2[0] = vec1.x + (interpolate * (vec2.x - vec1.x));
			v2[1] = vec1.y + (interpolate * (vec2.y - vec1.y));
			v2[2] = vec1.z + (interpolate * (vec2.z - vec1.z));

			vec1 = data->pointList->at(offset + data->triIndex[i].meshIndex[1]);
			vec2 = data->pointList->at(offset2 + data->triIndex[i].meshIndex[1]);
			v3[0] = vec1.x + (interpolate * (vec2.x - vec1.x));
			v3[1] = vec1.y + (interpolate * (vec2.y - vec1.y));
			v3[2] = vec1.z + (interpolate * (vec2.z - vec1.z));

			// Calculate the normal of the triangle
			this->calculateNormal(v1, v2, v3);

			// Render the triangle
			glTexCoord2f(data->st[data->triIndex[i].stIndex[0]].s, 1 - data->st[data->triIndex[i].stIndex[0]].t);
			glVertex3fv(v1);
			glTexCoord2f(data->st[data->triIndex[i].stIndex[2]].s, 1 - data->st[data->triIndex[i].stIndex[2]].t);
			glVertex3fv(v2);
			glTexCoord2f(data->st[data->triIndex[i].stIndex[1]].s, 1 - data->st[data->triIndex[i].stIndex[1]].t);
			glVertex3fv(v3);
		}
	glEnd();

	glBindTexture(GL_TEXTURE_2D, 0);
}

void * MD2Model::getHeader(FILE * filePtr)
{
	ModelHeader_t * header = (ModelHeader_t *)malloc(sizeof(ModelHeader_t));
	header->ident = this->readInt(filePtr);
	header->version = this->readInt(filePtr);
	header->skinWidth = this->readInt(filePtr);
	header->skinHeight = this->readInt(filePtr);
	header->frameSize = this->readInt(filePtr);
	header->numSkins = this->readInt(filePtr);
	header->numVertices = this->readInt(filePtr);
	header->numST = this->readInt(filePtr);
	header->numTris = this->readInt(filePtr);
	header->numGLcmds = this->readInt(filePtr);
	header->numFrames = this->readInt(filePtr);
	header->offsetSkins = this->readInt(filePtr);
	header->offsetST = this->readInt(filePtr);
	header->offsetTris = this->readInt(filePtr);
	header->offsetFrames = this->readInt(filePtr);
	header->offsetGLcmds = this->readInt(filePtr);
	header->offsetEnd = this->readInt(filePtr);
	return header;
}

void * MD2Model::getData(FILE * filePtr, void * headerData)
{
	ModelData_t * data;
	ModelHeader_t * header = (ModelHeader_t *)headerData;
	Frame_t frame;
	int i, j;

	data = (ModelData_t *)malloc(sizeof(ModelData_t));
	data->numFrames = header->numFrames;
	data->numVertices = header->numVertices;
	data->numST = header->numST;
	data->numTriangles = header->numTris;
	data->frameSize = header->frameSize;
	data->pointList = new vector<ModelPoint>();

	for (i = 0; i < (int)data->numFrames; i++) {		
		// Read frame
		fseek(filePtr, i * data->frameSize + header->offsetFrames, SEEK_SET);
		frame.scale[0] = readFloat(filePtr);
		frame.scale[1] = readFloat(filePtr);
		frame.scale[2] = readFloat(filePtr);
		frame.translate[0] = readFloat(filePtr);
		frame.translate[1] = readFloat(filePtr);
		frame.translate[2] = readFloat(filePtr);
		fread(&frame.name, sizeof(char), 16, filePtr);

		frame.fp = (FramePoint_t *)calloc(data->numVertices, sizeof(FramePoint_t));
		fread(frame.fp, sizeof(FramePoint_t), header->numVertices, filePtr);

		// Save the vertices
		for (j = 0; j < header->numVertices; j++) {
			ModelPoint point;
			point.x = (frame.scale[0] * frame.fp[j].v[0] + frame.translate[0]);
			point.y = (frame.scale[1] * frame.fp[j].v[1] + frame.translate[1]);
			point.z = (frame.scale[2] * frame.fp[j].v[2] + frame.translate[2]);
			data->pointList->push_back(point);
		}
		free(frame.fp);
	}

	// Load the textures
	data->st = (TexCoord_t *)malloc(data->numST * sizeof(TexCoord_t));
	fseek(filePtr, header->offsetST, SEEK_SET);
	for (i = 0; i < data->numST; i++) {
		data->st[i].s = (float)readShort(filePtr) / (float)header->skinWidth;
		data->st[i].t = (float)readShort(filePtr) / (float)header->skinHeight;
	}

	data->triIndex = (Mesh_t *)malloc(data->numFrames * data->numTriangles * sizeof(Mesh_t));
	fseek(filePtr, header->offsetTris, SEEK_SET);
	for (i = 0; i < data->numTriangles; i++) {
		data->triIndex[i].meshIndex[0] = readShort(filePtr);
		data->triIndex[i].meshIndex[1] = readShort(filePtr);
		data->triIndex[i].meshIndex[2] = readShort(filePtr);
		data->triIndex[i].stIndex[0] = readShort(filePtr);
		data->triIndex[i].stIndex[1] = readShort(filePtr);
		data->triIndex[i].stIndex[2] = readShort(filePtr);
	}

	return data;
}

void MD2Model::calculateNormal(float * f1, float * f2, float * f3)
{
	float a[3], b[3], result[3];
	float length;

	a[0] = f1[0] - f2[0];
	a[1] = f1[1] - f2[1];
	a[2] = f1[2] - f2[2];

	b[0] = f1[0] - f3[0];
	b[1] = f1[1] - f3[1];
	b[2] = f1[2] - f3[2];

	result[0] = a[1] * b[2] - b[1] * a[2];
	result[1] = b[0] * a[2] - a[0] * b[2];
	result[2] = a[0] * b[1] - b[0] * a[1];

	// Calculate the length of the normal
	length = (float)sqrt((result[0] * result[0]) + (result[1] * result[1]) + (result[2] * result[2]));

	// Normalize and specify the normal
	glNormal3f(result[0] / length, result[1] / length, result[2] / length);
}

unsigned int MD2Model::readInt(FILE * filePtr)
{
	unsigned int value;
	fread(&value, sizeof(unsigned int), 1, filePtr);
	return value;
}

unsigned short MD2Model::readShort(FILE * filePtr)
{
	unsigned short value;
	fread(&value, sizeof(unsigned short), 1, filePtr);
	return value;
}

float MD2Model::readFloat(FILE * filePtr)
{
	float value;
	fread(&value, sizeof(float), 1, filePtr);
	return value;
}

//returns positio
void MD2Model::getSurfaces(std::vector<Triangle *> * triangles, unsigned int frame){
	
	// no memory leak here
	for(int i = 0; i < (int)triangles->size(); i++){
		Triangle * t = triangles->at(i);
		//triangles->pop_front();
		delete t;
	}
	
	//clear vector
	triangles->clear();
	
	if (modelData == NULL) return;

	ModelPoint vector;
	ModelData_t * data = (ModelData_t *)modelData;
	float v1[3], v2[3], v3[3];
	int offset = frame * data->numVertices;
	int i;
	
	for (i = 0; i < data->numTriangles; i++) {
		vector = data->pointList->at(offset + data->triIndex[i].meshIndex[0]);
		v1[0] = vector.x;
		v1[1] = vector.y;
		v1[2] = vector.z;
		vector = data->pointList->at(offset + data->triIndex[i].meshIndex[2]);
		v2[0] = vector.x;
		v2[1] = vector.y;
		v2[2] = vector.z;
		vector = data->pointList->at(offset + data->triIndex[i].meshIndex[1]);
		v3[0] = vector.x;
		v3[1] = vector.y;
		v3[2] = vector.z;
		//printf("Triangle added");
		triangles->push_back(new Triangle(v1, v2, v3));
	}
}

void MD2Model::getSurfaces(TargaImage * positionMap, TargaImage * normalMap, unsigned int frame){
	//clear textures
	//normalMap->setAlpha(1);

	if (modelData == NULL) return;

	Triangle * triangle;
	unsigned char* pixel;
	int pixelIndex;

	ModelPoint vector;
	ModelData_t * data = (ModelData_t *)modelData;
	float v1[3], v2[3], v3[3];
	int offset = frame * data->numVertices;
	int i;
	
	for (i = 0; i < data->numTriangles; i++) {
		vector = data->pointList->at(offset + data->triIndex[i].meshIndex[0]);
		v1[0] = vector.x;
		v1[1] = vector.y;
		v1[2] = vector.z;
		vector = data->pointList->at(offset + data->triIndex[i].meshIndex[2]);
		v2[0] = vector.x;
		v2[1] = vector.y;
		v2[2] = vector.z;
		vector = data->pointList->at(offset + data->triIndex[i].meshIndex[1]);
		v3[0] = vector.x;
		v3[1] = vector.y;
		v3[2] = vector.z;
		
		//calculate surface information and pixel index
		triangle = new Triangle(v1, v2, v3);
		pixelIndex = i * 4;

		// set position and radius data
		pixel = positionMap->data + pixelIndex;				
		pixel[0] = (unsigned char)triangle->pos[0];	// RED
		pixel[1] = (unsigned char)triangle->pos[1];	// GREEN
		pixel[2] = (unsigned char)triangle->pos[2];	// BLUE
		pixel[3] = (unsigned char)triangle->radius;	// ALPHA
		
		// set normal data
		pixel = positionMap->data + pixelIndex;
		pixel[0] = (unsigned char)triangle->normal[0];	// RED
		pixel[1] = (unsigned char)triangle->normal[1];	// GREEN
		pixel[2] = (unsigned char)triangle->normal[2];	// BLUE
		pixel[3] = (unsigned char)0; // ALPHA

		delete triangle;
	}
}


int MD2Model::getSurfaces(GLuint * positionMap, GLuint * normalMap,
						   GLfloat positionBuffer[256][256][4], GLfloat normalBuffer[256][256][4],
						   unsigned int frame){
	if (modelData == NULL) return -1;

	Triangle * triangle;

	ModelPoint vector;
	ModelData_t * data = (ModelData_t *)modelData;
	float v1[3], v2[3], v3[3];
	int offset = frame * data->numVertices;
	int i;

	int w  = 256;
	int h = 256;
	int pxl_i, pxl_j;
	
	for (i = 0; i < data->numTriangles; i++) {
		vector = data->pointList->at(offset + data->triIndex[i].meshIndex[0]);
		v1[0] = vector.x;
		v1[1] = vector.y;
		v1[2] = vector.z;
		vector = data->pointList->at(offset + data->triIndex[i].meshIndex[2]);
		v2[0] = vector.x;
		v2[1] = vector.y;
		v2[2] = vector.z;
		vector = data->pointList->at(offset + data->triIndex[i].meshIndex[1]);
		v3[0] = vector.x;
		v3[1] = vector.y;
		v3[2] = vector.z;
		
		//calculate surface information and pixel index
		triangle = new Triangle(v1, v2, v3);
		
		pxl_i = i/w;
		pxl_j = i%h;
		//pxl_index = i * sizeof(GLfloat) * 4;

		positionBuffer[pxl_i][pxl_j][0] = triangle->pos[0] ;
        positionBuffer[pxl_i][pxl_j][1] = triangle->pos[1];
        positionBuffer[pxl_i][pxl_j][2] = triangle->pos[2];
        positionBuffer[pxl_i][pxl_j][3] = 0.0f;

        normalBuffer[pxl_i][pxl_j][0] = triangle->normal[0];
        normalBuffer[pxl_i][pxl_j][1] = triangle->normal[1];
        normalBuffer[pxl_i][pxl_j][2] = triangle->normal[2];
        normalBuffer[pxl_i][pxl_j][3] = triangle->area;
		
		/*if(i == 0){
		printf("Triangle%d (%d, %d)[%f,%f,%f,%f]\n", i, pxl_i, pxl_j,
						positionBuffer[pxl_i][pxl_j][0], positionBuffer[pxl_i][pxl_j][1],
						positionBuffer[pxl_i][pxl_j][2], positionBuffer[pxl_i][pxl_j][3]);
		}*/
		delete triangle;
	}
	
	//printf("Triangles Done\n");
	glPixelStorei(GL_UNPACK_ALIGNMENT, 1);

	glGenTextures(1, positionMap);
	glBindTexture(GL_TEXTURE_2D, *positionMap);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, w, h, 0, GL_RGBA, GL_FLOAT, (GLvoid *) positionBuffer);

	//printf("Position Map Done\n");

	glGenTextures(1, normalMap);
	glBindTexture(GL_TEXTURE_2D, *normalMap);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, w, h, 0, GL_RGBA, GL_FLOAT, (GLvoid *) normalBuffer);

	//printf("Normal Map Done\n");

	return data->numTriangles;
}

void MD2Model::render_ambient(std::vector<Triangle *> * triangles, unsigned int frame) {
	if (modelData == NULL) return;

	ModelPoint vector;
	ModelData_t * data = (ModelData_t *)modelData;
	float v1[3], v2[3], v3[3];
	int offset = frame * data->numVertices;
	int i;

	//if (strlen(texPath) > 0)
	//	glBindTexture(GL_TEXTURE_2D, fetchTexture(texPath, true, true));

    glBegin(GL_TRIANGLES);
		for (i = 0; i < data->numTriangles; i++) {
			vector = data->pointList->at(offset + data->triIndex[i].meshIndex[0]);
			v1[0] = vector.x;
			v1[1] = vector.y;
			v1[2] = vector.z;
			vector = data->pointList->at(offset + data->triIndex[i].meshIndex[2]);
			v2[0] = vector.x;
			v2[1] = vector.y;
			v2[2] = vector.z;
			vector = data->pointList->at(offset + data->triIndex[i].meshIndex[1]);
			v3[0] = vector.x;
			v3[1] = vector.y;
			v3[2] = vector.z;

			// Calculate the normal of the triangle
			this->calculateNormal(v1, v2, v3);

			// Render the triangle
			setMaterial(1.0 - ((Triangle *)triangles->at(i))->occlusion);
			glVertexAttrib1f(surfaceAttributeLoc, i);
			glTexCoord2f(data->st[data->triIndex[i].stIndex[0]].s, 1 - data->st[data->triIndex[i].stIndex[0]].t);
			glVertex3fv(v1);

			glVertexAttrib1f(surfaceAttributeLoc, i);
			glTexCoord2f(data->st[data->triIndex[i].stIndex[2]].s, 1 - data->st[data->triIndex[i].stIndex[2]].t);
			glVertex3fv(v2);

			glVertexAttrib1f(surfaceAttributeLoc, i);
			glTexCoord2f(data->st[data->triIndex[i].stIndex[1]].s, 1 - data->st[data->triIndex[i].stIndex[1]].t);
			glVertex3fv(v3);
		}
	glEnd();

	//glBindTexture(GL_TEXTURE_2D, 0);
}

void setMaterial(float val){
	//printf("occlusion: %f", val);
	GLfloat material_All[] = {val, val, val, 1.0f};

	glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT, material_All);
	glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, material_All);
	glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, material_All);
	glMaterialfv(GL_FRONT_AND_BACK, GL_EMISSION, material_All);
	//glMaterialf(GL_FRONT_AND_BACK, GL_SHININESS, material_Se);
}